const moment = require('moment-timezone');
const config = require('../config/config');

class BusinessHours {
  constructor() {
    this.timezone = config.businessHours.timezone;
    this.startTime = config.businessHours.start;
    this.endTime = config.businessHours.end;
    this.businessDays = config.businessHours.days;
  }

  /**
   * Check if within business hours
   * @returns {boolean}
   */
  isBusinessHours() {
    const now = moment.tz(this.timezone);
    const dayOfWeek = now.day(); // 0 = Sunday, 1 = Monday, ...

    // Check if it's a business day
    if (!this.businessDays.includes(dayOfWeek)) {
      return false;
    }

    // Check time
    const currentTime = now.format('HH:mm');
    
    // If end time is 00:00, treat as 23:59 (end of day)
    let endTime = this.endTime;
    if (endTime === '00:00') {
      endTime = '23:59';
    }
    
    return currentTime >= this.startTime && currentTime <= endTime;
  }

  /**
   * Get current time in configured timezone
   * @returns {moment.Moment}
   */
  getCurrentTime() {
    return moment.tz(this.timezone);
  }

  /**
   * Format current time
   * @returns {string}
   */
  getFormattedCurrentTime() {
    return this.getCurrentTime().format('DD/MM/YYYY HH:mm:ss');
  }

  /**
   * Get next business hours
   * @returns {string}
   */
  getNextBusinessHours() {
    let next = moment.tz(this.timezone);
    
    // If within hours today, return today's hours
    if (this.isBusinessHours()) {
      return `Today from ${this.startTime} to ${this.endTime}`;
    }

    // Search for next business day
    let daysChecked = 0;
    while (daysChecked < 7) {
      next.add(1, 'day');
      const dayOfWeek = next.day();
      
      if (this.businessDays.includes(dayOfWeek)) {
        const dayName = this.getDayName(dayOfWeek);
        return `${dayName} from ${this.startTime} to ${this.endTime}`;
      }
      
      daysChecked++;
    }

    return `${this.startTime} to ${this.endTime}`;
  }

  /**
   * Get day of week name
   * @param {number} dayOfWeek - 0-6 (Sunday to Saturday)
   * @returns {string}
   */
  getDayName(dayOfWeek) {
    const days = {
      0: 'Sunday',
      1: 'Monday',
      2: 'Tuesday',
      3: 'Wednesday',
      4: 'Thursday',
      5: 'Friday',
      6: 'Saturday'
    };
    return days[dayOfWeek] || 'Business day';
  }

  /**
   * Return formatted text of business days
   * @returns {string}
   */
  getBusinessDaysText() {
    const dayNames = this.businessDays.map(day => this.getDayName(day));
    
    if (dayNames.length === 0) {
      return 'No days';
    }
    
    if (dayNames.length === 1) {
      return dayNames[0];
    }
    
    if (dayNames.length === 2) {
      return dayNames.join(' and ');
    }
    
    const lastDay = dayNames.pop();
    return dayNames.join(', ') + ' and ' + lastDay;
  }

  /**
   * Generate out-of-hours message
   * @param {string} language - Language code
   * @returns {string}
   */
  getOutOfHoursMessage(language = 'en') {
    const messages = {
      pt: `🕐 *Horário de Atendimento*

Nosso horário de atendimento é de *${this.startTime} às ${this.endTime}* (GMT+2, Maputo).

Sua mensagem foi recebida e será atendida assim que possível.

📅 Próximo atendimento: ${this.getNextBusinessHours()}

Obrigado pela compreensão! 🙏`,

      en: `🕐 *Business Hours*

Our business hours are *${this.startTime} to ${this.endTime}* (GMT+2, Maputo).

Your message has been received and will be answered as soon as possible.

📅 Next available: ${this.getNextBusinessHours()}

Thank you for your understanding! 🙏`,

      es: `🕐 *Horario de Atención*

Nuestro horario de atención es de *${this.startTime} a ${this.endTime}* (GMT+2, Maputo).

Su mensaje ha sido recibido y será atendido lo antes posible.

📅 Próxima atención: ${this.getNextBusinessHours()}

¡Gracias por su comprensión! 🙏`,

      fr: `🕐 *Heures d'ouverture*

Nos heures d'ouverture sont de *${this.startTime} à ${this.endTime}* (GMT+2, Maputo).

Votre message a été reçu et sera traité dès que possible.

📅 Prochaine disponibilité: ${this.getNextBusinessHours()}

Merci de votre compréhension! 🙏`
    };

    return messages[language] || messages.pt;
  }

  /**
   * Calculate time until next business hours in minutes
   * @returns {number}
   */
  getMinutesUntilNextBusinessHours() {
    if (this.isBusinessHours()) {
      return 0;
    }

    const now = moment.tz(this.timezone);
    let next = now.clone();
    
    // Search for next business hours
    let daysChecked = 0;
    while (daysChecked < 7) {
      const dayOfWeek = next.day();
      
      if (this.businessDays.includes(dayOfWeek)) {
        const [startHour, startMinute] = this.startTime.split(':').map(Number);
        next.set({ hour: startHour, minute: startMinute, second: 0 });
        
        if (next.isAfter(now)) {
          return next.diff(now, 'minutes');
        }
      }
      
      next.add(1, 'day');
      daysChecked++;
    }

    return -1;
  }
}

module.exports = new BusinessHours();
